﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using VeteransAffairs.Registries.BusinessAHOBPR;
using VeteransAffairs.Registries.BusinessManagerAHOBPR.MilitaryInfoService;
using System.Diagnostics.Tracing;
using System.Diagnostics;

namespace VeteransAffairs.Registries.BusinessManagerAHOBPR.Emis
{
    public class DoDDeploymentImporter : AHOBPRBaseBO, IDeploymentImporter, IDisposable
    {
        private eMISMilitaryInformationSerivcePortTypesClient _client;

        private InputHeaderInfo _inputHeaderInfo;

        public DoDDeploymentImporter()
        {
            _client = new eMISMilitaryInformationSerivcePortTypesClient();
            _inputHeaderInfo = new InputHeaderInfo();
        }

        public DoDDeploymentImporter(eMISMilitaryInformationSerivcePortTypesClient client)
        {
            _client = client;
            _inputHeaderInfo = new InputHeaderInfo();
        }

        public InputHeaderInfo InputHeaderInfo
        {
            get { return _inputHeaderInfo; }
            set { _inputHeaderInfo = value; }
        }

        public BprDoDDeploymentImports GetEmisInfoForRegistrant(string edipi, string ssn, IRegistrantManager registrantManager)
        {
            //use the AHOBPRRegistrantManager to get info from DB on the Registrant
            if (registrantManager == null) { registrantManager = new AHOBPRRegistrantManager(); }
            REGISTRANT registrant = registrantManager.GetRegistrantById(edipi, ssn);
            // Create Hybrid Registrant Deployments object
            BprDoDDeploymentImports registrantWithDeployments = MergeRegistrantWithDoDDeploymentImport(registrant, edipi, ssn);
            //Get deployments from eMis
            List<MilitaryServiceEpisode> episodes = GetMilitaryServiceEpisodes(edipi);
            Dictionary<DateRange, string> branchesOfService = AssociateBranchesOfServiceWithDatesServed(episodes);
            eMISdeploymentResponseType deploymentResponse = GetDeployments(edipi);
            //add deployments to Hybrid Registrant Deployments Object
            try
            {
                registrantWithDeployments.deployments = SetRegistrantDeployments(deploymentResponse, branchesOfService);
            }
            catch (ArgumentNullException ex)
            {
                AHOBPRLogger.LogErrorMessage("Null Argument", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, ex.ParamName);
            }
            return registrantWithDeployments;
        }

        private BprDoDDeploymentImports MergeRegistrantWithDoDDeploymentImport(REGISTRANT registrant, string edipi, string ssn)
        {
            var registrantWithDeployments = new BprDoDDeploymentImports();
            registrantWithDeployments.edipi = edipi;
            registrantWithDeployments.ssn = ssn;
            if (registrant != null)
            {
                registrantWithDeployments._id = registrant.REGISTRANT_ID.ToString(CultureInfo.InvariantCulture);
                registrantWithDeployments.dob = registrant.BIRTH_DATE.HasValue ? Convert.ToDateTime(registrant.BIRTH_DATE).ToUniversalTime().ToString("yyyy'-'MM'-'dd'T'HH':'mm':'ss'.'fff'Z'", CultureInfo.InvariantCulture) : string.Empty;
                registrantWithDeployments.firstName = registrant.FIRST_NAME;
                registrantWithDeployments.importId = registrant.REGISTRANT_ID.ToString(CultureInfo.InvariantCulture);
                registrantWithDeployments.lastName = registrant.LAST_NAME;
                registrantWithDeployments.middleName = registrant.MIDDLE_NAME;
            }
            return registrantWithDeployments;
        }

        public BprDoDDeployment[] SetRegistrantDeployments(eMISdeploymentResponseType deploymentResponse, Dictionary<DateRange, string> branchesOfService)
        {
            if (deploymentResponse == null)
                throw new ArgumentNullException(nameof(deploymentResponse));
            if (branchesOfService == null)
                throw new ArgumentNullException(nameof(branchesOfService));

            List<BprDoDDeployment> deployments = new List<BprDoDDeployment>();
            //the array of deployments are from emis and may be null
            if (deploymentResponse.deployment != null)
            {
                foreach (Deployment emisDeployment in deploymentResponse.deployment)
                {
                    DeploymentData data = emisDeployment.deploymentData;
                    if (data.DeploymentLocation != null)
                    {
                        //Create a deployment object Need to do this for each location in the deployment
                        BprDoDDeployment bprDeployment;
                        //USE TOP LEVEL DEPLOYMENTDATA OBJECT INFO
                        foreach (DeploymentLocation location in data.DeploymentLocation)
                        {

                            string branchOfService = GetBranchOfService(location.deploymentLocationBeginDate,
                                                        location.deploymentLocationEndDate, branchesOfService);
                            try
                            {
                                bprDeployment = CreateDeployment(branchOfService, location);
                                deployments.Add(bprDeployment);
                            }
                            catch (ArgumentNullException ex)
                            {
                                AHOBPRLogger.LogErrorMessage("Null Argument", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, ex.ParamName);
                            }

                        }
                    }
                }
            }

            return deployments.ToArray();

        }

        public BprDoDDeployment CreateDeployment(string branchOfService, DeploymentLocation location)
        {
            if (location == null)
                throw new ArgumentNullException(nameof(location));

            var deployment = new BprDoDDeployment();
            deployment.dutyProjectCode = string.Empty;
            //A deploymentlocation will either have a country OR a major body of water code
            deployment.deploymentCountryCode = location.deploymentCountry ?? location.deploymentLocationMajorBodyOfWaterCode;
            DateRange deploymentDateRange = CreateNewDateRange(location.deploymentLocationBeginDate, location.deploymentLocationEndDate);
            deployment.deploymentStartDate = deploymentDateRange.Start.ToShortDateString();
            deployment.deploymentEndDate = deploymentDateRange.End.ToShortDateString();
            deployment.branchOfService = branchOfService;
            return deployment;
        }

        /// <summary>
        /// Get branch of service code based for a deployment based on service episdoe information.
        /// </summary>
        /// <param name="deploymentStart"></param>
        /// <param name="deploymentEnd"></param>
        /// <param name="branchesOfService"></param>
        /// <returns>branch of service code</returns>
        public string GetBranchOfService(DateTime deploymentStart, DateTime deploymentEnd, Dictionary<DateRange, string> branchesOfService)
        {
            var deploymentDateRange = new DateRange(deploymentStart, deploymentEnd);
            var epBranchOfServiceCode = branchesOfService.Keys.Where
                (
                //Check if any of the DateRange keys in branchesOfService include the deployment's DateRange
                e => e.Includes(deploymentDateRange)
                )
                //There's a match? Get that branchOfService code
                .Select(e => BranchOfServiceConverter.Convert(branchesOfService[e]))
                //If none of the branchesOfService DateRanges are a match, insert the default (Enums.BranchOfService.Unknown is 99)
                .DefaultIfEmpty("Unknown")
                //Take only the first, as the person should only be serving in one service at a time
                .First();
            return epBranchOfServiceCode;
        }

        /// <summary>
        /// Extract Branch of Service string from List of Military Episodes. If more than one episode is present, pick the last one
        /// </summary>
        /// <param name="episodes"></param>
        /// <returns></returns>
        public Dictionary<DateRange, string> AssociateBranchesOfServiceWithDatesServed(List<MilitaryServiceEpisode> episodes)
        {
            Dictionary<DateRange, string> branches = new Dictionary<DateRange, string>();
            //Check if there is more than one episode  present
            foreach (var episode in episodes)
            {
                DateRange dateRange = CreateNewDateRange(episode.militaryServiceEpisodeData.serviceEpisodeStartDate, episode.militaryServiceEpisodeData.serviceEpisodeEndDate);
                if (!branches.ContainsKey(dateRange))
                {
                    branches.Add(dateRange, episode.militaryServiceEpisodeData.branchOfServiceCode);
                }
            }
            return branches;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="militaryServiceEpisode"></param>
        /// <returns></returns>
        public DateRange CreateNewDateRange(MilitaryServiceEpisode militaryServiceEpisode)
        {
            //If they're active, it'll show up with a DateTime of 0/0/0001
            DateRange dateRange = CreateNewDateRange(militaryServiceEpisode.militaryServiceEpisodeData.serviceEpisodeStartDate, 
                                            militaryServiceEpisode.militaryServiceEpisodeData.serviceEpisodeEndDate);

            return dateRange;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="startDate"></param>
        /// <param name="endDate"></param>
        /// <returns></returns>
        public DateRange CreateNewDateRange(DateTime startDate, DateTime endDate)
        {
            //If they're active, it'll show up with a DateTime of 0/0/0001
            if (endDate.Day == 1 && endDate.Month == 1 &&  endDate.Year == 1)
            {
                //Add 500 to the startDate's year
                endDate = DateTime.Parse("9999-11-11");
            }
            return new DateRange(startDate, endDate);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="inputHeaderInfo"></param>
        /// <param name="inputEdipi"></param>
        /// <returns></returns>
        public eMISserviceEpisodeResponseType GetMilitaryServiceEpisodesResponse(InputHeaderInfo inputHeaderInfo, inputEdiPiOrIcn inputEdipi)
        {
            eMISserviceEpisodeResponseType response = null;
            try
            {
                response = _client.getMilitaryServiceEpisodes(ref inputHeaderInfo, inputEdipi);
            }
            catch (Exception ex)
            {
                Trace.WriteLine("\n" + DateTime.UtcNow + " ---  Exception: DoDDeploymentImporter.GetMilitaryServiceEpisodesResponse  " + ex.Message + "\n" + ex.StackTrace);
                AHOBPRLogger.LogErrorMessage("Exception", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, ex.StackTrace);
            }

            return response;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="edipi"></param>
        /// <returns></returns>
        public List<MilitaryServiceEpisode> GetMilitaryServiceEpisodes(string edipi)
        {
            var inputEdipi = InputEdipiOrIcnCreator.Create(edipi);
            eMISserviceEpisodeResponseType response = GetMilitaryServiceEpisodesResponse(_inputHeaderInfo, inputEdipi);
            return (response.militaryServiceEpisode != null) ? response.militaryServiceEpisode.ToList() : new List<MilitaryServiceEpisode>();
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="edipi"></param>
        /// <param name="ssn"></param>
        /// <returns></returns>
        public BprDoDDeploymentImports GetDoDDeploymentImports(string edipi, string ssn)
        {
            return GetEmisInfoForRegistrant(edipi, ssn, new AHOBPRRegistrantManager());
        }

        /// <summary>
        /// Overloaded method.Use the inputEdiPiOrIcn and InputHeaderInfo to create a getDeploymentRequest 
        /// </summary>
        /// <param name="edipi"></param>
        /// <param name="headerInfo"></param>
        /// <returns></returns>
        public eMISdeploymentResponseType GetDeployments(inputEdiPiOrIcn edipi, InputHeaderInfo headerInfo)
        {
            eMISdeploymentResponseType deployments = null;
            try
            {
                deployments = _client.getDeployment(ref headerInfo, edipi);
            }
            catch (Exception ex)
            {
                Trace.WriteLine("Exception: DoDDeploymentImporter.GetDeployments  " + ex.Message + "\n" + ex.StackTrace);
                AHOBPRLogger.LogErrorMessage("Exception", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, ex.StackTrace);
            }
            return deployments;
        }

        /// <summary>
        /// Overloaded method. Use this if you don't want to populate an InputHeaderInfo class
        /// </summary>
        /// <param name="edipi"></param>
        /// <returns></returns>
        public eMISdeploymentResponseType GetDeployments(inputEdiPiOrIcn edipi)
        {
            return GetDeployments(edipi, new InputHeaderInfo());
        }

        /// <summary>
        /// Just have a edipi string? Start here. It'll create the parameters needed to create a getDeploymentRequest
        /// </summary>
        /// <param name="edipi"></param>
        /// <returns></returns>
        public eMISdeploymentResponseType GetDeployments(string edipi)
        {
            //Because the client calls were generated and make not that much sense
            //we need to create an InputEdipiIcn which is used to create an inputEdiPiOrIcn
            //which can then be used to create a getDeploymentRequest in GetDeployments
            var inputEdipiIcn = InputEdipiOrIcnCreator.Create(edipi);
            return GetDeployments(inputEdipiIcn);
        }



        #region IDisposable Support
        private bool disposedValue; // To detect redundant calls

        protected virtual void Dispose(bool disposing)
        {
            if (!disposedValue)
            {
                if (disposing)
                {
                    // TODO: dispose managed state (managed objects).
                }

                // TODO: free unmanaged resources (unmanaged objects) and override a finalizer below.
                // TODO: set large fields to null.

                disposedValue = true;
            }
        }

        // This code added to correctly implement the disposable pattern.
        public void Dispose()
        {
            // Do not change this code. Put cleanup code in Dispose(bool disposing) above.
            Dispose(true);
            // TODO: uncomment the following line if the finalizer is overridden above.
            // GC.SuppressFinalize(this);
        }
        #endregion

    }
}
